#!/usr/bin/python 
#####################################################################################
#																					#
# 						Elastic HPC on GOOGLE COMPUTE ENGINE 						# 
#																					#
#####################################################################################
"""
	Elastic High Performance Computing on Google COmpute Engine version 0.1
	This library is for starting and initiating the Bioinformatics clusters 
	with the following tasks:

		1- Start cluster of instances on google compute engine.
		2- Add and remove instances from cluster.
		3- Delete cluster.
		4- Submitting PBS Torque Jobs.
		5- Running MPI run tasks.  
		6- Submiting SGE jobs
		7- Gluster File Sytem
		8- NFS.
	for more information please check this link: http://elastichpc.org/
"""
__author__ = 'ahmedaabdullwahed@gmail.com (Ahmed Abdullah)'

import os
import sys
import logging
import simplejson as json
import json
import time
import traceback
from apiclient.discovery import build
from apiclient.errors import HttpError
import httplib2
from httplib2 import HttpLib2Error
from oauth2client.client import AccessTokenRefreshError
from oauth2client.file import Storage
from oauth2client.tools import run
from oauth2client.client import flow_from_clientsecrets
import loadConfigFile as cf
import imp
DISK_TYPE	= 'PERSISTENT'

logger = imp.load_source('logger', '%s/%s'%(os.path.dirname(os.path.abspath(__file__)),'../logger.py'))


class ElasticHPCGCE():
	def __init__(self,project_id=None, CONFIG_FILE=None, CLIENT_SECRETS=None, STORAGE=None, API_VERSION=None, SCOPE=None, PROJECT_ID=None, command=True, LOGFILE=None):
		#self.config = cf.loadConfig('../config/cluster.conf')
		if CONFIG_FILE:
			SCOPE=[]
			configuration = cf.loadConfig(CONFIG_FILE)
			# Perform Authorization Flow
			working_dir, file = os.path.split(CONFIG_FILE) 
			CLIENT_SECRETS = '%s/../%s'%(os.path.dirname(os.path.abspath(__file__)), configuration['GCE']['client_secret'])
			if not os.path.exists(CLIENT_SECRETS):
				logger.log(LOGFILE, 'ERROR -- client secret does not exist')
				sys.exit(0)
			
			SCOPE.append(configuration['GCE']['compute_scope'])
			oauth2 = configuration['GCE']['oauth_storage']
			STORAGE			= '%s/%s'%(working_dir,oauth2)
			if not os.path.exists(STORAGE):
				logger.log(LOGFILE, 'ERROR -- authentication token does not exist, please upload oauth2.dat file')
				sys.exit(0)
			#STORAGE        = configuration['GCE']['oauth_storage']
			API_VERSION		= configuration['GCE']['api_version']
			PROJECT_ID		= configuration['GCE']['project_id']
			
			flow = flow_from_clientsecrets(CLIENT_SECRETS, scope=SCOPE)
			# Cloud Storage Authorization 
			storage 	= Storage(STORAGE)
			
			credentials = storage.get()
			if credentials is None or credentials.invalid:
				logger.log(LOGFILE, 'ERROR -- authentication token is invalid, try to upload the latest token')
				sys.exit(0)
				credentials = run(flow, storage)
			http 			= httplib2.Http()
			credentials.refresh(httplib2.Http())
			auth_http 		= credentials.authorize(http)
			
			self.service 	= build('compute', API_VERSION, http=auth_http)
			self.gce_url 	= 'https://www.googleapis.com/compute/%s/projects' % (API_VERSION)

			self.project_id = None
			if not project_id:
				self.project_id = PROJECT_ID
			else:
				self.project_id = project_id
			self.project_url = '%s/%s' % (self.gce_url, self.project_id)
		
		else:
			if (STORAGE and API_VERSION and PROJECT_ID): 
				storage = Storage(STORAGE)
				credentials = storage.get()
				if credentials is None or credentials.invalid:
					flow = flow_from_clientsecrets('%s/../config/client_secret.json'%(os.path.dirname(os.path.abspath(__file__))), scope=['https://www.googleapis.com/auth/devstorage.full_control'])
					credentials = run(flow, storage)
				http 			= httplib2.Http()
				credentials.refresh(httplib2.Http())
				auth_http 		= credentials.authorize(http)	
				self.service 	= build('compute', API_VERSION, http=auth_http)
				self.gce_url 	= 'https://www.googleapis.com/compute/%s/projects' % (API_VERSION)
				self.project_id = None
				if not project_id:
					self.project_id = PROJECT_ID
				else:
					self.project_id = project_id
				self.project_url = '%s/%s' % (self.gce_url, self.project_id)
			else:
				print "Error: please check the configuration input"
				sys.exit(0)
			
	# Create New instance:
	
	def start_instance(self,instance_name,disk_name,zone, machine_type=None,network=None, service_email=None, scopes=None, metadata=None,meta_value=None,startup_script=None,startup_script_url=None, blocking=True, fwname=None, fwprotocol=None, fwport=None, LOGFILE=None):
		
		if not instance_name:
			logger.log(LOGFILE, 'ERROR -- instance name is required')
			sys.exit(0)
			raise ValueError('instance_name required.')
		if not disk_name:
			logger.log(LOGFILE, 'ERROR -- disk_name is required')
			sys.exit(0)
			raise ValueError('disk_name required.')
		# Instance dictionary is sent in the body of the API request.
		instance = {}
		# Set required instance fields with defaults if not provided.
		instance['name'] = instance_name
		
		#if not zone:
		#	zone = self.config['compute']['zone']
		if not machine_type:
			machine_type = self.config['GCE']['machine_type']
		instance['machineType'] = '%s/zones/%s/machineTypes/%s' % (self.project_url, zone, machine_type)
		if not network:
			network = self.config['CLUSTER']['network']
		instance['networkInterfaces'] = [{'accessConfigs': [{'type': 'ONE_TO_ONE_NAT', 'name': 'External NAT'}],'network': '%s/global/networks/%s' % (self.project_url, network)}]
		# Make sure the disk exists, and apply disk to instance resource.
		disk_exists = self.get_disk(disk_name,zone, LOGFILE)
		if not disk_exists:
			raise DiskDoesNotExistError(disk_name + ' disk must exist.')
		instance['disks'] = [{'source': '%s/zones/%s/disks/%s' % (self.project_url, zone, disk_name),'boot': True,'type': DISK_TYPE}]
		# Set optional fields with provided values.
		if service_email and scopes:
			instance['serviceAccounts'] = [{'email': service_email, 'scopes': scopes}]
		# Set the instance metadata if provided.
		instance['metadata'] = {}
		instance['metadata']['items'] = []
		if metadata:
			#print "My SSH Key Meta-data"
			#print meta_value['value']
			instance['metadata']['items'].append(meta_value)
			
		# Set the instance startup script if provided.
		if startup_script:
			startup_script_resource = {'key': 'startup-script', 'value': open(startup_script, 'r').read()}
			#print startup_script_resource['value']
			instance['metadata']['items'].append(startup_script_resource)
		# Set the instance startup script URL if provided.
		if startup_script_url:
			startup_script_url_resource = {'key': 'startup-script-url', 'value': startup_script_url}
			instance['metadata']['items'].append(startup_script_url_resource)
		# Send the request.
		#print instance['metadata']['items']
		try:
			request = self.service.instances().insert(project=self.project_id, zone=zone, body=instance)
			response = self._execute_request(request)
			if response and blocking:
				response = self._blocking_call(response)

			if response and 'error' in response:
				logger.log(LOGFILE, "ERROR -- %s"%(response['error']['errors']))
				raise ApiOperationError(response['error']['errors'])
		except:
			logger.log(LOGFILE, "ERROR -- unable to create main node %s in project: %s"%(instance_name,self.project_id))
		if (fwname != None) and (fwprotocol != None) and (fwport != None):
			for i in range(len(fwname)):
				try:
					result = self.config_firewall(network,fwname[i],fwport[i],fwprotocol[i])
				except:
					logger.log(LOGFILE, "ERROR -- unable to add port %s, number %s , protocol %s to firewall configuration"%(fwname[i],fwport[i], fwprotocol[i]))
				#if not result:
				#	logging.info('firewall has error or port number is already exist please check config/cluster.conf')
		
		instanceResources = {}
		instanceResources = self.get_instance_resources(self.project_id,zone,instance_name,blocking=True)
		externalIP = instanceResources['networkInterfaces'][0]['accessConfigs'][0]['natIP']
		internalIP = instanceResources['networkInterfaces'][0]['networkIP']
		#logging.info("%s has External IP: %s and Internal IP : %s"%(instance_name, externalIP, internalIP))
		if externalIP:
			#logger.info("Cluster:'%s' Main node is running: %s"%(instance_name.split('node')[0],externalIP))
			return externalIP
		else:
			return None
	#configure firewall ports:
	def config_firewall(self,network,fwname,fwport,fwprotocol, blocking=True):
		allowed = []		
		firewall = {}
		firewall['name'] = fwname
		protocol = {}
		protocol ['IPProtocol'] = fwprotocol 
		protocol ['ports'] = [''+fwport+'']
		allowed.append(protocol)
		firewall['allowed'] = allowed
		firewall ["sourceRanges"] = ["0.0.0.0/0"]
		#logging.info("open firewall port number %s"%(protocol ['ports'][0]))
		firewall ['network'] = "%s/global/networks/%s" % (self.project_url, network)
		try:
			request = self.service.firewalls().insert(project=self.project_id, body=firewall)
			response = self._execute_request(request)
			if response and blocking:
				response = self._blocking_call(response)
			if response and 'error' in response:
				raise ApiOperationError(response['error']['errors'])
			return response
		except:
			#logging.info("Firewall port %s is already exist"%(protocol ['ports'][0]))
			return False	
	
	# get internal and external instance IPs:	
	def get_instance_resources(self, project_id,zone,instance_name,blocking=True):
		if not zone:
			zone = self.config['CLUSTER']['zone']
		request = self.service.instances().get(instance=instance_name, project=self.project_id, zone=zone)
		try:
			response = self._execute_request(request)
			
			return response
		except ApiError, e:
			return 
		
		
		

	# Get a disk:
	def get_disk(self, disk_name, zone, LOGFILE=None):
		#if not zone:
		#	zone = self.config['compute']['zone']
		request = self.service.disks().get(project=self.project_id, zone=zone, disk=disk_name)
		#logging.info(dict(request))
		try:
			response = self._execute_request(request)
			return response
		except ApiError, e:
			logger.log(LOGFILE, 'ERROR -- Main node hard disk %s does not exist'%(disk_name))
			return 'error'

	# Execute Requests:
	def _execute_request(self, request):
		try:
			response = request.execute()
		except AccessTokenRefreshError, e:
			logging.error('Access token is invalid.')
			raise ApiError(e)
		except HttpError, e:
			logging.error('Http response was not 2xx.')
			raise ApiError(e)
		except HttpLib2Error, e:
			logging.error('Transport error.')
			raise ApiError(e)
		except Exception, e:
			logging.error('Unexpected error occured.')
			traceback.print_stack()
			raise ApiError(e)
		return response
	# Create disk from blankDisk/Image/Snapshot
	def create_disk(self, disk_name, disk_size=None, project_id=None, zone=None,blocking=True, image=None, project=None, snapshot=None,LOGFILE=None):
		request = None
		disk = {}
		if not disk_name:
			raise ValueError('disk name required')
		disk['name'] = disk_name
		disk['mode'] = "READ_WRITE"
		if disk_size:	
			disk['sizeGb'] = disk_size
		#if not zone:
		#	zone = self.config['compute']['zone']
		if project:
			if image:
				source_image = '%s/%s/global/images/%s' % (self.gce_url, project, image)
				try:
					request = self.service.disks().insert(project=self.project_id, zone=zone, sourceImage=source_image, body=disk)
				except:
					logger.log(LOGFILE, 'ERROR -- unable to create hard disk from image-id: %s'%(image))
			if snapshot:
				sourceSnapshot = '%s/%s/global/snapshots/%s' %(self.gce_url, project, snapshot)
				disk['sourceSnapshot'] = sourceSnapshot
				try:
					request = self.service.disks().insert(project=self.project_id, zone=zone, body=disk)
				except:
					logger.log(LOGFILE, 'ERROR -- unable to create hard disk from snapshot: %s'%(snapshot))
		if (not image) and (not snapshot) and (not project):
			try:	
				if not disk_size or int(disk_size) < 200: 
					disk['sizeGb'] = "200"
			except:
				disk['sizeGb'] = "200"
			try:	
				request = self.service.disks().insert(project=self.project_id, zone=zone, body=disk)
			except:
				logger.log(LOGFILE, "ERROR -- unable to create empty hard disk with a size of %s" %(disk['sizeGb']))
		if request:
			response = self._execute_request(request)
			if response and blocking:
				response = self._blocking_call(response)
			if response and 'error' in response:
				raise ApiOperationError(response['error']['errors'])
			return response
		else:
			raise ValueError("Resource request has an error please check one of the following\n\t1- Snapshot\n\t2- Project\n\t3-image\ncheck cluster.conf configuration file ")
			return None
		
	# attache persistant disk to instance.
	def attache_disk(self, project_id, zone, instance, source_disk=None, disk_size=None, blocking=True):
		disk = {}
		if source_disk:
			disk['kind'] = "compute#attachedDisk"
			disk['type'] = "PERSISTENT"
			disk['mode'] = "READ_WRITE"
			disk['source'] = source_disk
			disk['autoDelete'] = False
			request = self.service.instances().attachDisk(project=project_id, zone=zone, instance=instance, body=disk)
			response = self._execute_request(request)
			if response and blocking:
				response = self._blocking_call(response)
			if response and 'error' in response:
				raise ApiOperationError(response['error']['errors'])
			return response
		else:
			print "create a new disk"
			if disk_size:
				"make the disk size as required min 200 GB"
			else:
				"defualt size is 200GB"
			return "ay 7aga now"

	# Block until the operation is complete.	
	def _blocking_call(self, response):

		status = response['status']
		
		while status != 'DONE' and response:
			operation_id = response['name']
		
			if 'zone' in response:
				zone = response['zone'].rsplit('/', 1)[-1]
				request = self.service.zoneOperations().get(project=self.project_id, zone=zone, operation=operation_id)
			
			else:
				request = self.service.globalOperations().get(project=self.project_id, operation=operation_id)
			response = self._execute_request(request)
			
			if response:
				status = response['status']
				#logging.info('Waiting until operation is DONE. Current status: %s', status)
				if status != 'DONE':
					time.sleep(3)
		return response
	  
	#list instances
	def list_instances(self, zone, list_filter=None):
		#if not zone:
		#	zone = self.config['compute']['zone']
		request = None
		if list_filter:
			request = self.service.instances().list(project=self.project_id, zone=zone, filter=list_filter)
		else:
			request = self.service.instances().list(project=self.project_id, zone=zone)
		
		response = self._execute_request(request)
		
		if response and 'items' in response:
			return response['items']
		return []

	# Stop Instances
	def delete_instance(self,instance_name,zone,blocking=True):
    
		if not instance_name:
			raise ValueError('instance_name required.')
			
		#if not zone:
		#	zone = self.config['compute']['zone']

		# Delete the instance.
		request = self.service.instances().delete(project=self.project_id, zone=zone, instance=instance_name)
		response = self._execute_request(request)
		if response and blocking:
			response = self._blocking_call(response)

		if response and 'error' in response:
			raise ApiOperationError(response['error']['errors'])
		return response

	def list_disks(self, zone,blocking=True):
		request = self.service.disks().list(project=self.project_id, zone=zone)
		response = self._execute_request(request)		
		if response and 'items' in response:
			return response['items']
		return []
		
	def delete_disk(self, disk_name, zone, blocking=True):
		
		if not disk_name:
			raise ValueError('disk_name required.')
		#if not zone:
		#	zone = self.config['compute']['zone']
		# Delete the disk.
		request = self.service.disks().delete(project=self.project_id, zone=zone, disk=disk_name)
		response = self._execute_request(request)
		if response and blocking:
			response = self._blocking_call(response)
		if response and 'error' in response:
			raise ApiOperationError(response['error']['errors'])
		return response
		
class Error(Exception):
	"""Base class for exceptions in this module."""
	pass


class ApiError(Error):
	"""Error occurred during API call."""
	pass


class ApiOperationError(Error):
	"""Raised when an API operation contains an error."""

	def __init__(self, error_list):
		"""Initialize the Error.
		Args:
			error_list: the list of errors from the operation.
		"""

		super(ApiOperationError, self).__init__()
		self.error_list = error_list

	def __str__(self):
		"""String representation of the error."""
		return repr(self.error_list)


class DiskDoesNotExistError(Error):
	"""Disk to be used for instance boot does not exist."""
	pass 
